// Copyright (c) 2015-2016 by Silicon Laboratories Inc.  All rights reserved.
// The program contained in this listing is proprietary to Silicon Laboratories,
// headquartered in Austin, Texas, U.S.A. and is subject to worldwide copyright
// protection, including protection under the United States Copyright Act of 1976
// as an unpublished work, pursuant to Section 104 and Section 408 of Title XVII
// of the United States code.  Unauthorized copying, adaptation, distribution,
// use, or display is prohibited by this law.

#include <string>
#include <sstream>
#include <iostream>
#include <vector>
#ifdef _WIN32
#include <Windows.h>
#pragma comment (lib, "SLABHIDtoSMBus.lib")
#else
#include "OsDep.h"
#endif
#include "stdio.h"
#include "SLABCP2112.h"
#include "util.h"
#include "smt.h"

const WORD g_LockAll =
        HID_SMBUS_LOCK_VID |
        HID_SMBUS_LOCK_PID |
        HID_SMBUS_LOCK_POWER |
        HID_SMBUS_LOCK_POWER_MODE |
        HID_SMBUS_LOCK_RELEASE_VERSION |
        HID_SMBUS_LOCK_MFG_STR |
        HID_SMBUS_LOCK_PRODUCT_STR |
        HID_SMBUS_LOCK_SERIAL_STR;

void AbortOnErr( HID_SMBUS_STATUS status, std::string funcName)
{
    if( status != HID_SMBUS_SUCCESS)
    {
        char msg[ 128];
        sprintf( msg, /*SIZEOF_ARRAY( msg),*/ "%s returned 0x%x", funcName.c_str(), status);
        throw CDllErr( msg);
    }
}

//---------------------------------------------------------------------------------
DWORD LibSpecificNumDevices( const CVidPid &oldVidPid, const CVidPid &newVidPid)
{
    DWORD newDevCnt;
    AbortOnErr( HidSmbus_GetNumDevices( &newDevCnt, oldVidPid.m_Vid, oldVidPid.m_Pid ), "HidSmbus_GetNumDevices");
    DWORD oldDevCnt = 0;
    if( oldVidPid.m_Vid != newVidPid.m_Vid || oldVidPid.m_Pid != newVidPid.m_Pid)
    {
        AbortOnErr( HidSmbus_GetNumDevices( &oldDevCnt, oldVidPid.m_Vid, oldVidPid.m_Pid ), "HidSmbus_GetNumDevices");
    }
    return newDevCnt + oldDevCnt;
}
//---------------------------------------------------------------------------------
class CCP2112Dev
{
public:
    CCP2112Dev( const CVidPid &FilterVidPid, DWORD devIndex);
    ~CCP2112Dev();
    HID_SMBUS_DEVICE  handle() const { return m_H; }
    bool              isLocked() const;
    void              lock() const;
    void              reset() const;
    CDevType          getDevType() const;
    CVidPid           getVidPid() const;
    BYTE              getPowerMode() const;
    BYTE              getMaxPower() const;
    WORD              getDevVer() const;
    std::vector<BYTE> getSerNum( bool isAscii) const;
    std::vector<BYTE> getManufacturer( bool isAscii) const;
    std::vector<BYTE> getProduct( bool isAscii) const;
    void              setSerNum( const std::vector<BYTE> &str, bool isAscii) const;
    void              setManufacturer( const std::vector<BYTE> &str, bool isAscii) const;
    void              setProduct( const std::vector<BYTE> &str, bool isAscii) const;
private:
    HID_SMBUS_DEVICE m_H;
};
CCP2112Dev::CCP2112Dev( const CVidPid &FilterVidPid, DWORD devIndex)
{
    AbortOnErr( HidSmbus_Open( &m_H, devIndex, FilterVidPid.m_Vid, FilterVidPid.m_Pid), "HidSmbus_Open");
}
CCP2112Dev::~CCP2112Dev()
{
    HID_SMBUS_STATUS status = HidSmbus_Close( m_H);
    if( status != HID_SMBUS_SUCCESS)
    {
        std::cerr << "HidSmbus_Close failed\n";
    }
}
bool CCP2112Dev::isLocked() const
{
    BYTE lock;
    AbortOnErr( HidSmbus_GetLock( m_H, &lock), "HidSmbus_GetLock");
    return (lock & g_LockAll) == 0;
}
void CCP2112Dev::lock() const
{
    AbortOnErr( HidSmbus_SetLock( m_H, g_LockAll), "HidSmbus_SetLock");
}
void  CCP2112Dev::reset() const
{
    AbortOnErr( HidSmbus_Reset( m_H), "HidSmbus_Reset");
}
CDevType CCP2112Dev::getDevType() const
{
    BYTE partNum;
    BYTE version;
    AbortOnErr( HidSmbus_GetPartNumber( m_H, &partNum, &version ), "HidSmbus_GetPartNumber");
    return CDevType( partNum);
}
CVidPid CCP2112Dev::getVidPid() const
{
    WORD vid; WORD pid; BYTE maxPower; BYTE powerMode; WORD devVer;
    AbortOnErr( HidSmbus_GetUsbConfig( m_H, &vid, &pid, &maxPower, &powerMode, &devVer), "HidSmbus_GetUsbConfig");
    return CVidPid( vid, pid);
}
BYTE CCP2112Dev::getPowerMode() const
{
    WORD vid; WORD pid; BYTE maxPower; BYTE powerMode; WORD devVer;
    AbortOnErr( HidSmbus_GetUsbConfig( m_H, &vid, &pid, &maxPower, &powerMode, &devVer), "HidSmbus_GetUsbConfig");
    return powerMode;
}
BYTE CCP2112Dev::getMaxPower() const
{
    WORD vid; WORD pid; BYTE maxPower; BYTE powerMode; WORD devVer;
    AbortOnErr( HidSmbus_GetUsbConfig( m_H, &vid, &pid, &maxPower, &powerMode, &devVer), "HidSmbus_GetUsbConfig");
    return maxPower;
}
WORD CCP2112Dev::getDevVer() const
{
    WORD vid; WORD pid; BYTE maxPower; BYTE powerMode; WORD devVer;
    AbortOnErr( HidSmbus_GetUsbConfig( m_H, &vid, &pid, &maxPower, &powerMode, &devVer), "HidSmbus_GetUsbConfig");
    return devVer;
}
std::vector<BYTE> CCP2112Dev::getSerNum( bool) const
{
    std::vector<BYTE> str( MAX_UCHAR);
    BYTE CchStr = 0;
    AbortOnErr( HidSmbus_GetSerialString( m_H, reinterpret_cast<char *>( str.data()), &CchStr), "HidSmbus_GetSerialString");
    str.resize( CchStr);
    return str;
}
std::vector<BYTE> CCP2112Dev::getManufacturer( bool) const
{
    std::vector<BYTE> str( MAX_UCHAR);
    BYTE CchStr = 0;
    AbortOnErr( HidSmbus_GetManufacturingString( m_H, reinterpret_cast<char *>( str.data()), &CchStr), "HidSmbus_GetManufacturingString");
    str.resize( CchStr);
    return str;
}
std::vector<BYTE> CCP2112Dev::getProduct( bool) const
{
    std::vector<BYTE> str( MAX_UCHAR);
    BYTE CchStr = 0;
    AbortOnErr( HidSmbus_GetProductString( m_H, reinterpret_cast<char *>( str.data()), &CchStr), "HidSmbus_GetProductString");
    str.resize( CchStr);
    return str;
}
void CCP2112Dev::setSerNum( const std::vector<BYTE> &str, bool) const
{
    AbortOnErr( HidSmbus_SetSerialString( m_H, reinterpret_cast<char *>( const_cast<BYTE *>( str.data())), static_cast<BYTE>( str.size())), "HidSmbus_SetSerialString");
}
void CCP2112Dev::setManufacturer( const std::vector<BYTE> &str, bool) const
{
    AbortOnErr( HidSmbus_SetManufacturingString( m_H, reinterpret_cast<char *>( const_cast<BYTE *>( str.data())), static_cast<BYTE>( str.size())), "HidSmbus_SetManufacturingString");
}
void CCP2112Dev::setProduct( const std::vector<BYTE> &str, bool) const
{
    AbortOnErr( HidSmbus_SetProductString( m_H, reinterpret_cast<char *>( const_cast<BYTE *>( str.data())), static_cast<BYTE>( str.size())), "HidSmbus_SetProductString");
}
//---------------------------------------------------------------------------------
//---------------------------------------------------------------------------------
struct CCP2112Parms : public CDevParms<CCP2112Dev>
{
    CCP2112Parms() : m_ManufStr( false /*supportsUnicode*/) {}
    virtual void readParm( const std::string &parmName);
    void program( const CCP2112Dev &dev, const std::vector<BYTE> *pSerNum) const;
    void verify( const CCP2112Dev &dev, CSerNumSet &serNumSet) const;
protected:
    CManufacturerString<CCP2112Dev>  m_ManufStr;
};
void CCP2112Parms::readParm( const std::string &parmName)
{
    if( m_ManufStr.readParm( parmName))
    {
        return;
    }
    CDevParms::readParm( parmName);
}
void CCP2112Parms::program( const CCP2112Dev &dev, const std::vector<BYTE> *pSerNum) const
{
    CDevParms::program( dev, pSerNum);
    m_ManufStr.program( dev);

    BYTE mask = 0;
    if( m_VidPidSpecified)
    {
        mask |= HID_SMBUS_SET_VID;
        mask |= HID_SMBUS_SET_PID;
    }
    if( m_PowerModeSpecified)
    {
        mask |= HID_SMBUS_SET_POWER_MODE;
    }
    if( m_MaxPowerSpecified)
    {
        mask |= HID_SMBUS_SET_POWER;
    }
    if( m_DevVerSpecified)
    {
        mask |= HID_SMBUS_SET_RELEASE_VERSION;
    }
    AbortOnErr( HidSmbus_SetUsbConfig( dev.handle(), m_Vid, m_Vid, m_MaxPower, m_PowerMode, m_DevVer, mask), "HidSmbus_SetUsbConfig");
}
void CCP2112Parms::verify( const CCP2112Dev &dev, CSerNumSet &serNumSet) const
{
    CDevParms::verify( dev, serNumSet);
    m_ManufStr.verify( dev);
}
//---------------------------------------------------------------------------------
void LibSpecificMain( const CDevType &devType, const CVidPid &vidPid, int argc, const char * argv[])
{
    if( devType.Value() == 0xc)
    {
        DevSpecificMain<CCP2112Dev,CCP2112Parms> ( devType, vidPid, argc, argv);
    }
    else
    {
        throw CSyntErr( "Unsupported PartNum");
    }
}
